/***************************************************************************

  machine.c

  Functions to emulate general aspects of the machine (RAM, ROM, interrupts,
  I/O ports)

  Tapper machine started by Chris Kirmse

  Hacked version with externs removed

***************************************************************************/

#include <stdio.h>

#include "driver.h"
#include "z80/z80.h"
#include "machine/z80fmly.h"
/*#include "machine/6821pia.h" */
#include "timer.h"


int pedal_sensitivity = 4;			/* Amount of change to read each time the pedal keys are pulsed */
int weighting_factor = 0;			/* Progressive weighting factor */

int mcr_loadnvram;

static unsigned char soundlatch[4];
static unsigned char soundstatus;

/* z80 ctc */
static void ctc_interrupt (int state)
{
        cpu_cause_interrupt (0, Z80_VECTOR(0,state) );
}

static z80ctc_interface ctc_intf =
{  
        1,                  /* 1 chip */
        { 0 },              /* clock (filled in from the CPU 0 clock */
        { 0 },              /* timer disables */
        { ctc_interrupt },  /* interrupt handler */ 
        { 0 },              /* ZC/TO0 callback */
        { 0 },              /* ZC/TO1 callback */
        { 0 }               /* ZC/TO2 callback */
};              

                
        
/* used for the sound boards */
static int dacval;
static int suspended;


/***************************************************************************

  Generic MCR handlers

***************************************************************************/

void mcr_init_machine(void)
{
   int i;

	/* reset the sound */
   for (i = 0; i < 4; i++)
      soundlatch[i] = 0;
   soundstatus = 0;
   suspended = 0;

   /* initialize the CTC */
   ctc_intf.baseclock[0] = Machine->drv->cpu[0].cpu_clock;
   z80ctc_init (&ctc_intf);

   /* daisy chain set */
	{
		static Z80_DaisyChain daisy_chain[] =
		{
			{ z80ctc_reset, z80ctc_interrupt, z80ctc_reti, 0 }, /* CTC number 0 */
			{ 0,0,0,-1}         /* end mark */
		};
		cpu_setdaisychain (0,daisy_chain );
	}

   /* can't load NVRAM right away */
   mcr_loadnvram = 0;
}

int mcr_interrupt (void)
{
	/* once per frame, pulse the CTC line 3 */
	z80ctc_0_trg3_w (0, 1);
	z80ctc_0_trg3_w (0, 0);
	return ignore_interrupt ();
}

void mcr_delayed_write (int param)
{
	soundlatch[param >> 8] = param & 0xff;
}

void mcr_writeport(int port,int value)
{
	switch (port)
	{
		case 0:	/* OP0  Write latch OP0 (coin meters, 2 led's and cocktail 'flip') */
		   return;

		case 4:	/* Write latch OP4 */
			return;

		case 0x1c:	/* WIRAM0 - write audio latch 0 */
		case 0x1d:	/* WIRAM0 - write audio latch 1 */
		case 0x1e:	/* WIRAM0 - write audio latch 2 */
		case 0x1f:	/* WIRAM0 - write audio latch 3 */
			timer_set (TIME_NOW, ((port - 0x1c) << 8) | (value & 0xff), mcr_delayed_write);
/*			mcr_delayed_write (((port - 0x1c) << 8) | (value & 0xff));*/
/*			soundlatch[port - 0x1c] = value;*/
			return;

		case 0xe0:	/* clear watchdog timer */
			watchdog_reset_w (0, 0);
			return;

		case 0xe8:
			/* A sequence of values gets written here at startup; we don't know why;
			   However, it does give us the opportunity to tweak the IX register before
			   it's checked in Tapper and Timber, thus eliminating the need for patches
			   The value 5 is written last; we key on that, and only modify IX if it is
			   currently 0; hopefully this is 99.9999% safe :-) */
			if (value == 5)
			{
				Z80_Regs temp;
				Z80_GetRegs (&temp);

	#ifndef USE_DRZ80
				if (temp.IX.D == 0)
					temp.IX.D += 1;
	#else
				if (temp.regs.Z80IX == 0)
					temp.regs.Z80IX+= 1;
	#endif

				Z80_SetRegs (&temp);
			}
			return;

		case 0xf0:	/* These are the ports of a Z80-CTC; it generates interrupts in mode 2 */
		case 0xf1:
		case 0xf2:
		case 0xf3:
		  z80ctc_0_w (port - 0xf0, value);
		  return;
	}

}


int mcr_readport(int port)
{
	/* ports 0-4 are read directly via the input ports structure */
	port += 5;

   /* only a few ports here */
   switch (port)
   {
		case 0x07:	/* Read audio status register */

			/* once the system starts checking the sound, memory tests are done; load the NVRAM */
		   mcr_loadnvram = 1;
			return soundstatus;

		case 0x10:	/* Tron reads this as an alias to port 0 -- does this apply to all ports 10-14? */
			return cpu_readport (port & 0x0f);
   }

	return 0;
}


void mcr_soundstatus_w (int offset,int data)
{
   soundstatus = data;
}


int mcr_soundlatch_r (int offset)
{
   return soundlatch[offset];
}



/***************************************************************************

  Game-specific input handlers

**************************************************************************/

/* Kozmik Krooz'r -- dial reader */
int kroozr_dial_r(int offset)
{
	int dial = readinputport (7);
	int val = readinputport (1);

	val |= (dial & 0x80) >> 1;
	val |= (dial & 0x70) >> 4;

	return val;
}


/* Kozmik Krooz'r -- joystick readers */
int kroozr_trakball_x_r(int data)
{
	int val = readinputport (6);

	if (val & 0x02)		/* left */
		return 0x64 - 0x34;
	if (val & 0x01)		/* right */
		return 0x64 + 0x34;
	return 0x64;
}

int kroozr_trakball_y_r(int data)
{
	int val = readinputport (6);

	if (val & 0x08)		/* up */
		return 0x64 - 0x34;
	if (val & 0x04)		/* down */
		return 0x64 + 0x34;
	return 0x64;
}
